/*
 * Decompiled with CFR 0.152.
 */
package org.openecard.scio;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import javax.annotation.Nonnull;
import javax.smartcardio.CardException;
import javax.smartcardio.CardTerminal;
import javax.smartcardio.CardTerminals;
import org.openecard.common.ifd.scio.NoSuchTerminal;
import org.openecard.common.ifd.scio.SCIOErrorCode;
import org.openecard.common.ifd.scio.SCIOException;
import org.openecard.common.ifd.scio.SCIOTerminal;
import org.openecard.common.ifd.scio.SCIOTerminals;
import org.openecard.common.ifd.scio.TerminalState;
import org.openecard.common.ifd.scio.TerminalWatcher;
import org.openecard.common.util.Pair;
import org.openecard.scio.PCSCExceptionExtractor;
import org.openecard.scio.PCSCFactory;
import org.openecard.scio.PCSCTerminal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PCSCTerminals
implements SCIOTerminals {
    private static final Logger LOG = LoggerFactory.getLogger(PCSCTerminals.class);
    private static final long WAIT_DELTA = 1500L;
    private final PCSCFactory terminalFactory;
    private CardTerminals terminals;

    PCSCTerminals(@Nonnull PCSCFactory terminalFactory) {
        this.terminalFactory = terminalFactory;
        this.loadTerminals();
    }

    private void reloadFactory() {
        this.terminalFactory.reloadPCSC();
        this.loadTerminals();
    }

    private void loadTerminals() {
        this.terminals = this.terminalFactory.getRawFactory().terminals();
    }

    @Override
    public List<SCIOTerminal> list() throws SCIOException {
        return this.list(SCIOTerminals.State.ALL);
    }

    @Override
    public List<SCIOTerminal> list(SCIOTerminals.State state) throws SCIOException {
        return this.list(state, true);
    }

    public List<SCIOTerminal> list(SCIOTerminals.State state, boolean firstTry) throws SCIOException {
        LOG.trace("Entering list().");
        try {
            CardTerminals.State scState = this.convertState(state);
            List<CardTerminal> scList = this.terminals.list(scState);
            ArrayList<SCIOTerminal> list = this.convertTerminals(scList);
            LOG.trace("Leaving list().");
            return Collections.unmodifiableList(list);
        }
        catch (CardException ex) {
            SCIOErrorCode code = PCSCExceptionExtractor.getCode(ex);
            if (code == SCIOErrorCode.SCARD_E_NO_READERS_AVAILABLE) {
                LOG.debug("No reader available exception.");
                return Collections.emptyList();
            }
            if (code == SCIOErrorCode.SCARD_E_NO_SERVICE || code == SCIOErrorCode.SCARD_E_SERVICE_STOPPED) {
                if (firstTry) {
                    LOG.debug("No service available exception, reloading PCSC and trying again.");
                    this.reloadFactory();
                    return this.list(state, false);
                }
                LOG.debug("No service available exception, returning empty list.");
                return Collections.emptyList();
            }
            String msg = "Failed to retrieve list from terminals instance.";
            LOG.error(msg, ex);
            throw new SCIOException(msg, code, ex);
        }
    }

    private CardTerminals.State convertState(@Nonnull SCIOTerminals.State state) {
        switch (state) {
            case ALL: {
                return CardTerminals.State.ALL;
            }
            case CARD_PRESENT: {
                return CardTerminals.State.CARD_PRESENT;
            }
            case CARD_ABSENT: {
                return CardTerminals.State.CARD_ABSENT;
            }
        }
        LOG.error("Unknown state type requested: {}", (Object)state);
        throw new IllegalArgumentException("Invalid state type requested.");
    }

    private SCIOTerminal convertTerminal(@Nonnull CardTerminal scTerminal) {
        return new PCSCTerminal(scTerminal);
    }

    private ArrayList<SCIOTerminal> convertTerminals(List<CardTerminal> terminals) {
        ArrayList<SCIOTerminal> result = new ArrayList<SCIOTerminal>(terminals.size());
        for (CardTerminal t : terminals) {
            result.add(this.convertTerminal(t));
        }
        return result;
    }

    @Override
    public SCIOTerminal getTerminal(@Nonnull String name) throws NoSuchTerminal {
        CardTerminal t = this.terminals.getTerminal(name);
        if (t == null) {
            throw new NoSuchTerminal(String.format("Terminal '%s' does not exist in the system.", name));
        }
        return this.convertTerminal(t);
    }

    @Override
    public TerminalWatcher getWatcher() throws SCIOException {
        return new PCSCWatcher(this);
    }

    private static class PCSCWatcher
    implements TerminalWatcher {
        private final PCSCTerminals parent;
        private final PCSCTerminals own;
        private Queue<TerminalWatcher.StateChangeEvent> pendingEvents;
        private Collection<String> terminals;
        private Collection<String> cardPresent;

        public PCSCWatcher(@Nonnull PCSCTerminals parent) {
            this.parent = parent;
            this.own = new PCSCTerminals(parent.terminalFactory);
        }

        @Override
        public SCIOTerminals getTerminals() {
            return this.parent;
        }

        @Override
        public List<TerminalState> start() throws SCIOException {
            LOG.trace("Entering start().");
            if (this.pendingEvents != null) {
                throw new IllegalStateException("Trying to initialize already initialized watcher instance.");
            }
            this.pendingEvents = new LinkedList<TerminalWatcher.StateChangeEvent>();
            this.terminals = new HashSet<String>();
            this.cardPresent = new HashSet<String>();
            try {
                this.own.terminals.waitForChange(1L);
                List<CardTerminal> javaTerminals = this.own.terminals.list();
                ArrayList<TerminalState> result = new ArrayList<TerminalState>(javaTerminals.size());
                LOG.debug("Detecting initial terminal status.");
                for (CardTerminal next : javaTerminals) {
                    String name = next.getName();
                    boolean cardInserted = next.isCardPresent();
                    LOG.debug("Terminal='{}' cardPresent={}", (Object)name, (Object)cardInserted);
                    this.terminals.add(name);
                    if (cardInserted) {
                        this.cardPresent.add(name);
                        result.add(new TerminalState(name, true));
                        continue;
                    }
                    result.add(new TerminalState(name, false));
                }
                LOG.trace("Leaving start() with {} states.", (Object)result.size());
                return Collections.unmodifiableList(result);
            }
            catch (CardException ex) {
                SCIOErrorCode code = PCSCExceptionExtractor.getCode(ex);
                if (code == SCIOErrorCode.SCARD_E_NO_READERS_AVAILABLE) {
                    LOG.debug("No reader available exception.");
                    return Collections.emptyList();
                }
                if (code == SCIOErrorCode.SCARD_E_NO_SERVICE || code == SCIOErrorCode.SCARD_E_SERVICE_STOPPED) {
                    LOG.debug("No service available exception, reloading PCSC and returning empty list.");
                    this.parent.reloadFactory();
                    this.own.loadTerminals();
                    return Collections.emptyList();
                }
                String msg = "Failed to retrieve status from the PCSC system.";
                LOG.error(msg, ex);
                throw new SCIOException(msg, code, ex);
            }
            catch (IllegalStateException ex) {
                LOG.debug("No reader available exception.");
                return Collections.emptyList();
            }
        }

        @Override
        public TerminalWatcher.StateChangeEvent waitForChange() throws SCIOException {
            return this.waitForChange(0L);
        }

        @Override
        public TerminalWatcher.StateChangeEvent waitForChange(long timeout) throws SCIOException {
            LOG.trace("Entering waitForChange() with timeout={}.", (Object)timeout);
            if (this.pendingEvents == null) {
                throw new IllegalStateException("Calling wait on uninitialized watcher instance.");
            }
            if (timeout == 0L) {
                timeout = Long.MAX_VALUE;
            }
            while (timeout > 0L) {
                Pair<Boolean, Boolean> waitResult;
                long startTime = System.nanoTime();
                TerminalWatcher.StateChangeEvent nextEvent = this.pendingEvents.poll();
                if (nextEvent != null) {
                    LOG.trace("Leaving waitForChange() with queued event.");
                    return nextEvent;
                }
                try {
                    waitResult = this.internalWait(timeout);
                }
                catch (CardException ex) {
                    String msg = "Error while waiting for a state change in the terminals.";
                    LOG.error(msg, ex);
                    throw new SCIOException(msg, PCSCExceptionExtractor.getCode(ex), ex);
                }
                boolean changed = (Boolean)waitResult.p1;
                boolean error = (Boolean)waitResult.p2;
                if (!changed) {
                    LOG.trace("Leaving waitForChange() with no event.");
                    return new TerminalWatcher.StateChangeEvent();
                }
                HashSet<String> newTerminals = new HashSet<String>();
                HashSet<String> newCardPresent = new HashSet<String>();
                if (!error) {
                    try {
                        List<CardTerminal> newStates = this.own.terminals.list();
                        for (CardTerminal next : newStates) {
                            String name = next.getName();
                            newTerminals.add(name);
                            if (!next.isCardPresent()) continue;
                            newCardPresent.add(name);
                        }
                    }
                    catch (CardException ex) {
                        String msg = "Failed to retrieve status of the observed terminals.";
                        LOG.error(msg, ex);
                        throw new SCIOException(msg, PCSCExceptionExtractor.getCode(ex), ex);
                    }
                }
                Collection<String> cardRemoved = PCSCWatcher.subtract(this.cardPresent, newCardPresent);
                Collection<TerminalWatcher.StateChangeEvent> crEvents = PCSCWatcher.createEvents(TerminalWatcher.EventType.CARD_REMOVED, cardRemoved);
                Collection<String> termRemoved = PCSCWatcher.subtract(this.terminals, newTerminals);
                Collection<TerminalWatcher.StateChangeEvent> trEvents = PCSCWatcher.createEvents(TerminalWatcher.EventType.TERMINAL_REMOVED, termRemoved);
                Collection<String> termAdded = PCSCWatcher.subtract(newTerminals, this.terminals);
                Collection<TerminalWatcher.StateChangeEvent> taEvents = PCSCWatcher.createEvents(TerminalWatcher.EventType.TERMINAL_ADDED, termAdded);
                Collection<String> cardAdded = PCSCWatcher.subtract(newCardPresent, this.cardPresent);
                Collection<TerminalWatcher.StateChangeEvent> caEvents = PCSCWatcher.createEvents(TerminalWatcher.EventType.CARD_INSERTED, cardAdded);
                this.terminals = newTerminals;
                this.cardPresent = newCardPresent;
                this.pendingEvents.addAll(crEvents);
                this.pendingEvents.addAll(trEvents);
                this.pendingEvents.addAll(taEvents);
                this.pendingEvents.addAll(caEvents);
                if (!this.pendingEvents.isEmpty()) {
                    LOG.trace("Leaving waitForChange() with fresh event.");
                    return this.pendingEvents.remove();
                }
                long finishTime = System.nanoTime();
                long delta = finishTime - startTime;
                LOG.trace("Start wait loop again with reduced timeout value ({} ms).", (Object)(timeout -= delta / 1000000L));
            }
            LOG.trace("Leaving waitForChange() with no event.");
            return new TerminalWatcher.StateChangeEvent();
        }

        private void sleep(long millis) throws SCIOException {
            try {
                Thread.sleep(millis);
            }
            catch (InterruptedException ex2) {
                String msg = "Wait interrupted by another thread.";
                throw new SCIOException(msg, SCIOErrorCode.SCARD_E_SERVICE_STOPPED);
            }
        }

        private Pair<Boolean, Boolean> internalWait(long timeout) throws CardException, SCIOException {
            if (timeout < 0L) {
                throw new IllegalArgumentException("Negative timeout value given.");
            }
            if (timeout == 0L) {
                timeout = Long.MAX_VALUE;
            }
            block7: while (true) {
                long waitTime;
                if (timeout == 0L) {
                    return new Pair<Boolean, Boolean>(false, false);
                }
                if (timeout < 1500L) {
                    waitTime = timeout;
                    timeout = 0L;
                } else {
                    timeout -= 1500L;
                    waitTime = 1500L;
                }
                try {
                    boolean change = this.own.terminals.waitForChange(1L);
                    if (change) {
                        return new Pair<Boolean, Boolean>(true, false);
                    }
                    this.sleep(waitTime);
                    change = this.own.terminals.waitForChange(1L);
                    if (change) {
                        return new Pair<Boolean, Boolean>(true, false);
                    }
                }
                catch (CardException ex) {
                    switch (PCSCExceptionExtractor.getCode(ex)) {
                        case SCARD_E_NO_SERVICE: 
                        case SCARD_E_SERVICE_STOPPED: {
                            LOG.debug("No service available exception, reloading PCSC.");
                            this.parent.reloadFactory();
                            this.own.loadTerminals();
                        }
                        case SCARD_E_NO_READERS_AVAILABLE: {
                            if (!this.terminals.isEmpty()) {
                                return new Pair<Boolean, Boolean>(true, true);
                            }
                            LOG.debug("Waiting for PCSC system to become available again.");
                            this.sleep(waitTime);
                            continue block7;
                        }
                    }
                    throw ex;
                }
                catch (IllegalStateException ex) {
                    if (!this.terminals.isEmpty()) {
                        return new Pair<Boolean, Boolean>(true, true);
                    }
                    LOG.debug("Waiting for PCSC system to become available again.");
                    this.sleep(waitTime);
                    continue;
                }
                ArrayList<CardTerminal> currentTerms = new ArrayList<CardTerminal>(this.own.terminals.list());
                if (currentTerms.size() != this.terminals.size()) {
                    return new Pair<Boolean, Boolean>(true, false);
                }
                HashSet<String> newTermNames = new HashSet<String>();
                for (CardTerminal next : currentTerms) {
                    newTermNames.add(next.getName());
                }
                int sizeBefore = newTermNames.size();
                if (sizeBefore != this.terminals.size()) {
                    return new Pair<Boolean, Boolean>(false, false);
                }
                newTermNames.addAll(this.terminals);
                int sizeAfter = newTermNames.size();
                if (sizeBefore != sizeAfter) break;
            }
            return new Pair<Boolean, Boolean>(false, false);
        }

        private static <T> Collection<T> subtract(Collection<T> a, Collection<T> b) {
            HashSet<T> result = new HashSet<T>(a);
            result.removeAll(b);
            return result;
        }

        private static Collection<TerminalWatcher.StateChangeEvent> createEvents(TerminalWatcher.EventType type, Collection<String> list) {
            ArrayList<TerminalWatcher.StateChangeEvent> result = new ArrayList<TerminalWatcher.StateChangeEvent>(list.size());
            for (String next : list) {
                result.add(new TerminalWatcher.StateChangeEvent(type, next));
            }
            return result;
        }
    }
}

