/*
 * Decompiled with CFR 0.152.
 */
package org.openecard.transport.httpcore.cookies;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.openecard.transport.httpcore.cookies.Cookie;
import org.openecard.transport.httpcore.cookies.CookieException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CookieManager {
    private static final Logger logger = LoggerFactory.getLogger(CookieManager.class);
    private static final Pattern cookiePat = Pattern.compile("(?<name>.+?)=(?<value>.+?)((; Path=(?<path>.+?))|(; Domain=(?<domain>.+?))|(; (?<httponly>HttpOnly))|(; (?<secure>Secure))|(; Expires=(?<expires>.+?))|(; Max-[Aa]ge=(?<maxage>\\d+?)))*", 66);
    private final int maxCookiesPerDomain = 50;
    private final int maxCookies = 3000;
    private final Map<String, List<Cookie>> cookieMap;
    private int currentCookieCount = 0;

    public CookieManager() {
        this.cookieMap = new HashMap<String, List<Cookie>>();
    }

    public CookieManager(@Nonnull Map<String, List<Cookie>> cookieMap) {
        this.cookieMap = cookieMap;
    }

    public void addCookie(@Nonnull String domain, @Nonnull String cookieHeaderValue) throws CookieException {
        if (this.currentCookieCount == 3000) {
            String msg = "The cookie storage is full.";
            logger.error(msg);
            throw new CookieException(msg);
        }
        Cookie cookie = this.createCookie(cookieHeaderValue);
        String domainKey = null;
        try {
            domainKey = this.createDomainKey(domain, cookie);
        }
        catch (MalformedURLException ex) {
            String msg = "Invalid value (%s) in the \"domain\" parameter received.";
            msg = String.format(msg, domain);
            logger.error(msg, ex);
            throw new CookieException(msg, ex);
        }
        if (!this.update(domainKey, cookie)) {
            List<Cookie> cookies = this.cookieMap.get(domainKey);
            if (cookies == null) {
                cookies = new ArrayList<Cookie>();
            }
            if (cookies.size() == 50) {
                String msg = "Maximal number of cookies per domain reached.";
                logger.error(msg);
                throw new CookieException(msg);
            }
            if (logger.isDebugEnabled()) {
                String msg = "Setting cookie %s for key %s.";
                msg = String.format(msg, cookieHeaderValue, domainKey);
                logger.debug(msg);
            }
            cookies.add(cookie);
            this.cookieMap.put(domainKey, cookies);
            ++this.currentCookieCount;
        }
    }

    @Nonnull
    private Cookie createCookie(@Nonnull String cookieHeaderValue) throws CookieException {
        Cookie cookie;
        Matcher matcher = cookiePat.matcher(cookieHeaderValue);
        boolean matches = matcher.matches();
        if (matches) {
            cookie = new Cookie(matcher.group("name"), matcher.group("value"));
            String secure = matcher.group("secure");
            String httpOnly = matcher.group("httponly");
            String maxage = matcher.group("maxage");
            String domain = matcher.group("domain");
            String path = matcher.group("path");
            String expires = matcher.group("expires");
            if (secure != null) {
                cookie.setSecure(true);
            }
            if (httpOnly != null) {
                cookie.setHttpOnly(true);
            }
            if (maxage != null) {
                cookie.setMaxAge(maxage);
            }
            if (domain != null) {
                cookie.setDomain(domain);
            }
            if (path != null) {
                cookie.setPath(path);
            }
            if (expires != null) {
                cookie.setExpires(expires);
            }
        } else {
            String msg = "The cookie \"%s\" is invalid and will not be stored.";
            msg = String.format(msg, cookieHeaderValue);
            throw new CookieException(msg);
        }
        return cookie;
    }

    public void deleteAllCookies() {
        this.cookieMap.clear();
        this.currentCookieCount = 0;
    }

    public void deleteCookie(@Nonnull String domain, @Nonnull String name) throws CookieException {
        URL url;
        try {
            url = new URL(domain);
        }
        catch (MalformedURLException ex) {
            String msg = "The \"domain\" parameter contains an invalid URL.";
            throw new CookieException(msg, ex);
        }
        List<Cookie> cookies = this.cookieMap.get(url.getHost());
        if (cookies != null && !cookies.isEmpty()) {
            for (int i = 0; i < cookies.size(); ++i) {
                Cookie c = cookies.get(i);
                if (!c.getName().equals(name)) continue;
                cookies.remove(i);
                --this.currentCookieCount;
                break;
            }
        }
    }

    @Nullable
    public String getCookieHeaderValue(@Nonnull String domain) throws CookieException {
        try {
            Set<String> keySet = this.cookieMap.keySet();
            ArrayList<Cookie> usableCookies = new ArrayList<Cookie>();
            URL domAsURL = new URL(domain);
            for (String string : keySet) {
                if (!domAsURL.getHost().endsWith(string)) continue;
                List<Cookie> domainCookies = this.cookieMap.get(string);
                ArrayList<Cookie> cleanList = new ArrayList<Cookie>();
                if (domainCookies != null && domainCookies.size() > 0) {
                    for (Cookie c : domainCookies) {
                        if (c.getDomain().isEmpty()) {
                            if (!string.equals(domAsURL.getHost())) continue;
                            if (c.getPath().isEmpty()) {
                                usableCookies.add(c);
                                cleanList.add(c);
                                continue;
                            }
                            if (!domAsURL.getPath().startsWith(c.getPath())) continue;
                            usableCookies.add(c);
                            cleanList.add(c);
                            continue;
                        }
                        if (!c.getPath().isEmpty() && !domAsURL.getPath().startsWith(c.getPath())) continue;
                        usableCookies.add(c);
                        cleanList.add(c);
                    }
                }
                for (Cookie c : cleanList) {
                    this.clean(string, c);
                }
            }
            StringBuilder headerValue = new StringBuilder();
            for (Cookie c : usableCookies) {
                headerValue.append(c.getName());
                headerValue.append("=");
                headerValue.append(c.getValue());
                headerValue.append("; ");
            }
            if (headerValue.length() > 0) {
                int n = headerValue.lastIndexOf("; ");
                headerValue.delete(n, n + 2);
                return headerValue.toString();
            }
        }
        catch (MalformedURLException ex) {
            String msg = "The given value (%s) of the \"domain\" parameter is not valid URL.";
            msg = String.format(msg, domain);
            logger.error(msg, ex);
            throw new CookieException(msg, ex);
        }
        return null;
    }

    @Nonnull
    private String createDomainKey(@Nonnull String domain, @Nonnull Cookie cookie) throws MalformedURLException {
        URL url = new URL(domain);
        String domainAttr = cookie.getDomain();
        if (!domainAttr.isEmpty()) {
            if (domainAttr.startsWith(".")) {
                domainAttr = domainAttr.substring(1);
            }
            return domainAttr;
        }
        return url.getHost();
    }

    private void clean(@Nonnull String key, @Nonnull Cookie c) {
        if (c.isExpired()) {
            List<Cookie> domainCookies = this.cookieMap.get(key);
            domainCookies.remove(c);
            --this.currentCookieCount;
            if (domainCookies.isEmpty()) {
                this.cookieMap.remove(key);
            }
        }
    }

    private boolean update(String domainKey, Cookie cookie) {
        List<Cookie> cookies = this.cookieMap.get(domainKey);
        if (cookies == null || cookies.isEmpty()) {
            return false;
        }
        for (Cookie c : cookies) {
            if (!c.equals(cookie)) continue;
            cookie.setCreationTime(c.getCreationTime());
            this.clean(domainKey, c);
            if (!cookie.isExpired()) {
                cookies.add(cookie);
            }
            return true;
        }
        return false;
    }
}

