/*
 * Decompiled with CFR 0.152.
 */
package org.openecard.common.util;

import iso.std.iso_iec._24727.tech.schema.InputAPDUInfoType;
import iso.std.iso_iec._24727.tech.schema.PasswordAttributesType;
import iso.std.iso_iec._24727.tech.schema.PasswordTypeType;
import iso.std.iso_iec._24727.tech.schema.Transmit;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;
import org.openecard.common.util.ByteUtils;
import org.openecard.common.util.UtilException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PINUtils {
    private static final Logger LOG = LoggerFactory.getLogger(PINUtils.class);

    public static Transmit buildVerifyTransmit(char[] rawPIN, PasswordAttributesType attributes, byte[] template, byte[] slotHandle) throws UtilException {
        byte[] pin = PINUtils.encodePin(rawPIN, attributes);
        byte[] pinCmd = ByteUtils.concatenate(template, (byte)pin.length);
        pinCmd = ByteUtils.concatenate(pinCmd, pin);
        Arrays.fill(pin, (byte)0);
        Transmit transmit = new Transmit();
        transmit.setSlotHandle(slotHandle);
        InputAPDUInfoType pinApdu = new InputAPDUInfoType();
        pinApdu.setInputAPDU(pinCmd);
        pinApdu.getAcceptableStatusCode().add(new byte[]{-112, 0});
        transmit.getInputAPDUInfo().add(pinApdu);
        return transmit;
    }

    public static byte[] encodePin(char[] rawPin, PasswordAttributesType attributes) throws UtilException {
        PasswordTypeType pwdType = attributes.getPwdType();
        int minLen = attributes.getMinLength().intValue();
        int maxLen = attributes.getMaxLength() == null ? 0 : attributes.getMaxLength().intValue();
        int storedLen = attributes.getStoredLength().intValue();
        boolean needsPadding = PINUtils.needsPadding(attributes);
        byte padChar = PINUtils.getPadChar(attributes, needsPadding);
        String encoding = "UTF-8";
        try {
            switch (pwdType) {
                case ASCII_NUMERIC: {
                    encoding = "US-ASCII";
                }
                case UTF_8: {
                    byte[] textPin = PINUtils.encodeTextPin(encoding, rawPin, minLen, storedLen, maxLen, needsPadding, padChar);
                    return textPin;
                }
                case ISO_9564_1: 
                case BCD: 
                case HALF_NIBBLE_BCD: {
                    byte[] bcdPin = PINUtils.encodeBcdPin(pwdType, rawPin, minLen, storedLen, maxLen, needsPadding, padChar);
                    return bcdPin;
                }
            }
            String msg = "Unsupported PIN encoding requested.";
            UtilException ex = new UtilException("http://www.bsi.bund.de/ecard/api/1.1/resultminor/ifdl/IO#unknownPINFormat", msg);
            LOG.error(ex.getMessage(), ex);
            throw ex;
        }
        catch (UnsupportedEncodingException ex) {
            throw new UtilException(ex);
        }
        catch (IOException ex) {
            throw new UtilException(ex);
        }
    }

    public static byte[] createPinMask(PasswordAttributesType attributes) throws UtilException {
        PasswordTypeType pwdType = attributes.getPwdType();
        int minLen = attributes.getMinLength().intValue();
        int maxLen = attributes.getMaxLength() == null ? 0 : attributes.getMaxLength().intValue();
        int storedLen = attributes.getStoredLength().intValue();
        boolean needsPadding = PINUtils.needsPadding(attributes);
        if (!needsPadding) {
            return new byte[0];
        }
        byte padChar = PINUtils.getPadChar(attributes, needsPadding);
        if (storedLen <= 0) {
            throw new UtilException("PIN mask can only be created when storage size is known.");
        }
        if (PasswordTypeType.HALF_NIBBLE_BCD == pwdType) {
            padChar = (byte)(padChar | 0xF0);
        }
        byte[] mask = new byte[storedLen];
        Arrays.fill(mask, padChar);
        if (PasswordTypeType.ISO_9564_1 == pwdType) {
            mask[0] = 32;
        }
        return mask;
    }

    public static byte[] encodeTextPin(String encoding, char[] rawPin, int minLen, int storedLen, int maxLen, boolean needsPadding, byte padChar) throws UnsupportedEncodingException, UtilException {
        if (needsPadding && storedLen <= 0) {
            String msg = "Padding is required, but no stored length is given.";
            throw new UtilException(msg);
        }
        if (rawPin.length < minLen) {
            String msg = String.format("Entered PIN is too short, enter at least %d characters.", minLen);
            throw new UtilException(msg);
        }
        if (maxLen > 0 && rawPin.length > maxLen) {
            String msg = String.format("Entered PIN is too long, enter at most %d characters.", maxLen);
            throw new UtilException(msg);
        }
        Charset charset = Charset.forName(encoding);
        ByteBuffer bb = charset.encode(CharBuffer.wrap(rawPin));
        byte[] pinBytes = new byte[bb.remaining()];
        bb.get(pinBytes);
        if (bb.hasArray()) {
            Arrays.fill(bb.array(), (byte)0);
        }
        if (storedLen > 0 && pinBytes.length > storedLen) {
            Arrays.fill(pinBytes, (byte)0);
            String msg = String.format("Storage size for PIN exceeded, only %d bytes are allowed.", storedLen);
            throw new UtilException(msg);
        }
        if (needsPadding && pinBytes.length < storedLen) {
            int missingBytes = storedLen - pinBytes.length;
            byte[] filler = new byte[missingBytes];
            Arrays.fill(filler, padChar);
            byte[] pinBytesTmp = ByteUtils.concatenate(pinBytes, filler);
            Arrays.fill(pinBytes, (byte)0);
            pinBytes = pinBytesTmp;
        }
        return pinBytes;
    }

    public static byte[] encodeBcdPin(PasswordTypeType pwdType, char[] rawPin, int minLen, int storedLen, int maxLen, boolean needsPadding, byte padChar) throws UtilException, IOException {
        int i;
        ByteArrayOutputStream o = new ByteArrayOutputStream();
        int pinSize = rawPin.length;
        if (PasswordTypeType.ISO_9564_1 == pwdType) {
            byte head = (byte)(0x20 | 0xF & pinSize);
            o.write(head);
        }
        if (PasswordTypeType.HALF_NIBBLE_BCD == pwdType) {
            for (i = 0; i < pinSize; ++i) {
                char nextChar = rawPin[i];
                byte digit = (byte)(0xF0 | PINUtils.getByte(nextChar));
                o.write(digit);
            }
        } else if (PasswordTypeType.BCD == pwdType || PasswordTypeType.ISO_9564_1 == pwdType) {
            for (i = 0; i < pinSize; i += 2) {
                byte b1 = (byte)(PINUtils.getByte(rawPin[i]) << 4);
                byte b2 = (byte)(padChar & 0xF);
                if (i + 1 < pinSize) {
                    b2 = (byte)(PINUtils.getByte(rawPin[i + 1]) & 0xF);
                }
                byte b = (byte)(b1 | b2);
                o.write(b);
            }
        }
        if (needsPadding && o.size() < storedLen) {
            int missingBytes = storedLen - o.size();
            byte[] filler = new byte[missingBytes];
            Arrays.fill(filler, padChar);
            o.write(filler);
        }
        return o.toByteArray();
    }

    private static byte getByte(char c) throws UtilException {
        if (c >= '0' && c <= '9') {
            return (byte)(c - 48);
        }
        UtilException ex = new UtilException("Entered PIN contains invalid characters.");
        LOG.error(ex.getMessage(), ex);
        throw ex;
    }

    private static byte getPadChar(PasswordAttributesType attributes, boolean needsPadding) throws UtilException {
        if (PasswordTypeType.ISO_9564_1.equals((Object)attributes.getPwdType())) {
            return -1;
        }
        byte[] padChars = attributes.getPadChar();
        if (padChars != null && padChars.length == 1) {
            return padChars[0];
        }
        if (needsPadding) {
            UtilException ex = new UtilException("Unsupported combination of PIN parameters concerning padding.");
            throw ex;
        }
        return 0;
    }

    private static boolean needsPadding(PasswordAttributesType attributes) {
        PasswordTypeType pwdType = attributes.getPwdType();
        if (PasswordTypeType.ISO_9564_1 == pwdType) {
            return true;
        }
        boolean needsPadding = attributes.getPwdFlags().contains("needs-padding");
        return needsPadding;
    }
}

