/*
 * Decompiled with CFR 0.152.
 */
package org.openecard.bouncycastle.crypto.tls;

import java.io.IOException;
import java.security.SecureRandom;
import org.openecard.bouncycastle.crypto.BlockCipher;
import org.openecard.bouncycastle.crypto.Digest;
import org.openecard.bouncycastle.crypto.params.KeyParameter;
import org.openecard.bouncycastle.crypto.params.ParametersWithIV;
import org.openecard.bouncycastle.crypto.tls.ProtocolVersion;
import org.openecard.bouncycastle.crypto.tls.TlsCipher;
import org.openecard.bouncycastle.crypto.tls.TlsContext;
import org.openecard.bouncycastle.crypto.tls.TlsFatalAlert;
import org.openecard.bouncycastle.crypto.tls.TlsMac;
import org.openecard.bouncycastle.crypto.tls.TlsUtils;
import org.openecard.bouncycastle.util.Arrays;

public class TlsBlockCipher
implements TlsCipher {
    protected TlsContext context;
    protected byte[] randomData;
    protected boolean useExplicitIV;
    protected BlockCipher encryptCipher;
    protected BlockCipher decryptCipher;
    protected TlsMac writeMac;
    protected TlsMac readMac;

    public TlsMac getWriteMac() {
        return this.writeMac;
    }

    public TlsMac getReadMac() {
        return this.readMac;
    }

    public TlsBlockCipher(TlsContext context, BlockCipher clientWriteCipher, BlockCipher serverWriteCipher, Digest clientWriteDigest, Digest serverWriteDigest, int cipherKeySize) throws IOException {
        ParametersWithIV decryptParams;
        ParametersWithIV encryptParams;
        byte[] server_write_IV;
        byte[] client_write_IV;
        this.context = context;
        this.randomData = new byte[256];
        context.getSecureRandom().nextBytes(this.randomData);
        this.useExplicitIV = ProtocolVersion.TLSv11.isEqualOrEarlierVersionOf(context.getServerVersion().getEquivalentTLSVersion());
        int key_block_size = 2 * cipherKeySize + clientWriteDigest.getDigestSize() + serverWriteDigest.getDigestSize();
        if (!this.useExplicitIV) {
            key_block_size += clientWriteCipher.getBlockSize() + serverWriteCipher.getBlockSize();
        }
        byte[] key_block = TlsUtils.calculateKeyBlock(context, key_block_size);
        int offset = 0;
        TlsMac clientWriteMac = new TlsMac(context, clientWriteDigest, key_block, offset, clientWriteDigest.getDigestSize());
        TlsMac serverWriteMac = new TlsMac(context, serverWriteDigest, key_block, offset += clientWriteDigest.getDigestSize(), serverWriteDigest.getDigestSize());
        KeyParameter client_write_key = new KeyParameter(key_block, offset += serverWriteDigest.getDigestSize(), cipherKeySize);
        KeyParameter server_write_key = new KeyParameter(key_block, offset += cipherKeySize, cipherKeySize);
        offset += cipherKeySize;
        if (this.useExplicitIV) {
            client_write_IV = new byte[clientWriteCipher.getBlockSize()];
            server_write_IV = new byte[serverWriteCipher.getBlockSize()];
        } else {
            client_write_IV = Arrays.copyOfRange(key_block, offset, offset + clientWriteCipher.getBlockSize());
            server_write_IV = Arrays.copyOfRange(key_block, offset += clientWriteCipher.getBlockSize(), offset + serverWriteCipher.getBlockSize());
            offset += serverWriteCipher.getBlockSize();
        }
        if (offset != key_block_size) {
            throw new TlsFatalAlert(80);
        }
        if (context.isServer()) {
            this.writeMac = serverWriteMac;
            this.readMac = clientWriteMac;
            this.encryptCipher = serverWriteCipher;
            this.decryptCipher = clientWriteCipher;
            encryptParams = new ParametersWithIV(server_write_key, server_write_IV);
            decryptParams = new ParametersWithIV(client_write_key, client_write_IV);
        } else {
            this.writeMac = clientWriteMac;
            this.readMac = serverWriteMac;
            this.encryptCipher = clientWriteCipher;
            this.decryptCipher = serverWriteCipher;
            encryptParams = new ParametersWithIV(client_write_key, client_write_IV);
            decryptParams = new ParametersWithIV(server_write_key, server_write_IV);
        }
        this.encryptCipher.init(true, encryptParams);
        this.decryptCipher.init(false, decryptParams);
    }

    @Override
    public int getPlaintextLimit(int ciphertextLimit) {
        int blockSize = this.encryptCipher.getBlockSize();
        int macSize = this.writeMac.getSize();
        int result = ciphertextLimit - ciphertextLimit % blockSize - macSize - 1;
        if (this.useExplicitIV) {
            result -= blockSize;
        }
        return result;
    }

    @Override
    public byte[] encodePlaintext(long seqNo, short type, byte[] plaintext, int offset, int len) {
        int i;
        int blockSize = this.encryptCipher.getBlockSize();
        int macSize = this.writeMac.getSize();
        ProtocolVersion version = this.context.getServerVersion();
        int padding_length = blockSize - 1 - (len + macSize) % blockSize;
        if (!version.isDTLS() && !version.isSSL()) {
            int maxExtraPadBlocks = (255 - padding_length) / blockSize;
            int actualExtraPadBlocks = this.chooseExtraPadBlocks(this.context.getSecureRandom(), maxExtraPadBlocks);
            padding_length += actualExtraPadBlocks * blockSize;
        }
        int totalSize = len + macSize + padding_length + 1;
        if (this.useExplicitIV) {
            totalSize += blockSize;
        }
        byte[] outbuf = new byte[totalSize];
        int outOff = 0;
        if (this.useExplicitIV) {
            byte[] explicitIV = new byte[blockSize];
            this.context.getSecureRandom().nextBytes(explicitIV);
            this.encryptCipher.init(true, new ParametersWithIV(null, explicitIV));
            System.arraycopy(explicitIV, 0, outbuf, outOff, blockSize);
            outOff += blockSize;
        }
        byte[] mac = this.writeMac.calculateMac(seqNo, type, plaintext, offset, len);
        System.arraycopy(plaintext, offset, outbuf, outOff, len);
        System.arraycopy(mac, 0, outbuf, outOff + len, mac.length);
        int padOffset = outOff + len + mac.length;
        for (i = 0; i <= padding_length; ++i) {
            outbuf[i + padOffset] = (byte)padding_length;
        }
        for (i = outOff; i < totalSize; i += blockSize) {
            this.encryptCipher.processBlock(outbuf, i, outbuf, i);
        }
        return outbuf;
    }

    @Override
    public byte[] decodeCiphertext(long seqNo, short type, byte[] ciphertext, int offset, int len) throws IOException {
        boolean badMac;
        int blockSize = this.decryptCipher.getBlockSize();
        int macSize = this.readMac.getSize();
        int minLen = Math.max(blockSize, macSize + 1);
        if (this.useExplicitIV) {
            minLen += blockSize;
        }
        if (len < minLen) {
            throw new TlsFatalAlert(50);
        }
        if (len % blockSize != 0) {
            throw new TlsFatalAlert(21);
        }
        if (this.useExplicitIV) {
            this.decryptCipher.init(false, new ParametersWithIV(null, ciphertext, offset, blockSize));
            offset += blockSize;
            len -= blockSize;
        }
        for (int i = 0; i < len; i += blockSize) {
            this.decryptCipher.processBlock(ciphertext, offset + i, ciphertext, offset + i);
        }
        int totalPad = this.checkPaddingConstantTime(ciphertext, offset, len, blockSize, macSize);
        int macInputLen = len - totalPad - macSize;
        byte[] decryptedMac = Arrays.copyOfRange(ciphertext, offset + macInputLen, offset + macInputLen + macSize);
        byte[] calculatedMac = this.readMac.calculateMacConstantTime(seqNo, type, ciphertext, offset, macInputLen, len - macSize, this.randomData);
        boolean bl = badMac = !Arrays.constantTimeAreEqual(calculatedMac, decryptedMac);
        if (badMac || totalPad == 0) {
            throw new TlsFatalAlert(20);
        }
        return Arrays.copyOfRange(ciphertext, offset, offset + macInputLen);
    }

    protected int checkPaddingConstantTime(byte[] buf, int off, int len, int blockSize, int macSize) {
        int end = off + len;
        byte lastByte = buf[end - 1];
        int padlen = lastByte & 0xFF;
        int totalPad = padlen + 1;
        int dummyIndex = 0;
        int padDiff = 0;
        if (this.context.getServerVersion().isSSL() && totalPad > blockSize || macSize + totalPad > len) {
            totalPad = 0;
        } else {
            int padPos = end - totalPad;
            do {
                padDiff = (byte)(padDiff | buf[padPos++] ^ lastByte);
            } while (padPos < end);
            dummyIndex = totalPad;
            if (padDiff != 0) {
                totalPad = 0;
            }
        }
        byte[] dummyPad = this.randomData;
        while (dummyIndex < 256) {
            padDiff = (byte)(padDiff | dummyPad[dummyIndex++] ^ lastByte);
        }
        dummyPad[0] = (byte)(dummyPad[0] ^ padDiff);
        return totalPad;
    }

    protected int chooseExtraPadBlocks(SecureRandom r, int max) {
        int x = r.nextInt();
        int n = this.lowestBitSet(x);
        return Math.min(n, max);
    }

    protected int lowestBitSet(int x) {
        if (x == 0) {
            return 32;
        }
        int n = 0;
        while ((x & 1) == 0) {
            ++n;
            x >>= 1;
        }
        return n;
    }
}

